<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Validator;
use \Exception;
use App\Helpers\CommonHelper;
use App\Models\Category;
use App\Models\Vendor;
use App\Models\Item;
use App\Models\itemImage;
use App\Models\productType;
use App\Models\Review;
use App\State;
use App\Models\UserDevice;
use App\Models\ProductDeliveryCharge;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Support\Carbon;
use App\Jobs\ProductVerificationJob;
use App\Models\Authors;
use Auth;
use DB;

class AdminItemController extends Controller
{
    /**
     * Http response data
     */
    private $data;

    /**
     * Http response code
     */
    private $code;

    /**
     * Response status
     */
    private $success;

    /**
     * Response message
     */
    private $message;

    /**
     * Init class attributes
     */
    public function __construct(Request $request)
    {
        $this->data     = new \stdClass();
        $this->code     = Response::HTTP_OK;
        $this->success  = true;
        $this->message  = '';
    }

    /**
     * get all user category list
     * @author  SOORYA
     */
    public function create(Request $request)
    {
        $this->data  =  [
            'categories'        => (new Category())->getCategories(),
            'vendors'           => (new Vendor())->getVendors(),
            'product_types'     => (new productType())->getProductTypes(),
            'parent_items'      => (new Item())->getListItem(),
            'authors'           => (new Authors())->getAuthorsList()

        ];
        return view('admin.products.create-product', $this->data);
    }

    public function editProduct(Request $request, $uid)
    {
        $product                = (new Item)->getItemByUid($uid);
        $selected_categories    = DB::table('category_item')
            ->join('categories', 'category_item.category_id', '=', 'categories.id')
            ->select('category_item.category_id', 'categories.name as category_name')
            ->where('category_item.item_id', $product->id)
            ->get();
        $selected_product_types = DB::table('item_product_type')
            ->join('product_types', 'item_product_type.product_type_id', '=', 'product_types.id')
            ->select('item_product_type.product_type_id', 'product_types.name as product_type_name')
            ->where('item_product_type.item_id', $product->id)
            ->get();
        $this->data  =  [
            'product'                   =>  $product,
            'categories'                => (new Category())->getExcludedCategories($selected_categories),
            'vendors'                   => (new Vendor())->getVendors(),
            'product_types'             => (new productType())->getExcludedProductTypes($selected_product_types),
            'selected_categories'       =>  $selected_categories,
            'selected_product_types'    =>  $selected_product_types,
            'parent_items'              => (new Item())->getListItem()

        ];
        return view('admin.products.edit-product', $this->data);
    }


    public function updateProduct(Request $request)
    {
        $item = (new Item)->getItemByUid($request->uid);
        $input =  [
            'name'                  => $request->name,
            'sku'                  => $request->sku,
            'descriptions'          => strip_tags($request->descriptions),
            'specification'         => $request->specification,
            'features'              => $request->features,
            'price'                 => str_replace([',', '₹'], '', $request->price),
            'mrp'                   => str_replace([',', '₹'], '', $request->mrp),
            'vendor_id'             => $request->vendor_id,
            'stock'                 => $request->stock,
            'color'                 => $request->color,
            'size'                  => $request->size,
            'parent_id'             => $request->parent_id,
            'slug'                  => $this->slugCreateUniq($request->slug, $item->id),


        ];

        $file                       = $request->file('product_image');
        if ($file) {
            $extension                  = $file->extension();
            $file_name                  = strtolower(str_replace(' ', '_', $request->name)) . '_' . time();
            $path                       = 'images/product/';
            Storage::disk('local')->putFileAs('public/' . $path, $file, $file_name . '.' . $extension);
            $input['image']             = '/storage/' . $path . $file_name . '.' . $extension;
        }
        (new Item())->updateItem($item->id, $input);
        $item->categories()->sync($request->category_id);
        $item->product_types()->sync($request->product_type_id);
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Product list successfully retrieved',
            'data'      => $this->data
        ], $this->code);
    }

    /**
     * 
     * create slug
     * 
     */
    public function slugCreateUniq($slug, $excludeId = null)
    {
        $originalSlug   = $slug;
        $count          = 1;
        while (Item::where('slug', $slug)
            ->when($excludeId, function ($query, $excludeId) {
                return $query->where('id', '!=', $excludeId);
            })
            ->exists()
        ) {
            $slug = $originalSlug . '-' . $count;
            $count++;
        }
        return $slug;
    }


    /**
     * 
     * @author  PMS
     */
    public function createNewItem(Request $request)
    {
        $input =  [
            'name'                  => $request->name,
            'sku'                   => strip_tags($request->sku),
            'descriptions'          => strip_tags($request->descriptions),
            'specification'         => $request->specification,
            'features'              => $request->features,
            'price'                 => str_replace([',', '₹'], '', $request->price),
            'mrp'                   => str_replace([',', '₹'], '', $request->mrp),
            'vendor_id'             => $request->vendor_id,
            'stock'                 => $request->stock,
            'color'                 => $request->color,
            'size'                  => $request->size,
            'parent_id'             => $request->parent_id,
            'slug'                  => $this->slugCreateUniq($request->slug),
            'is_draft'              => Carbon::now()->format('Y-m-d H:i:s'),
            'is_verified'           => Carbon::now()->format('Y-m-d H:i:s'),
         
        ];
        $file                       = $request->file('product_image');
        $extension                  = $file->extension();
        $file_name                  = strtolower(str_replace(' ', '_', $request->name)) . '_' . time();
        $path                       = 'images/product/';
        Storage::disk('local')->putFileAs('public/' . $path, $file, $file_name . '.' . $extension);
        $input['image']             = '/storage/' . $path . $file_name . '.' . $extension;
        $item =  (new Item())->createItem($input);
        $item->categories()->attach($request->category_id);
        $item->product_types()->attach($request->product_type_id);
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Product list successfully retrieved',
            'data'      => $this->data
        ], $this->code);
    }

    /**
     * get all user category list
     * @author  SOORYA
     */
    public function getItemList(Request $request)
    {
        $result = (new Item())->getAdminItemList($request);
        $this->data  = [
            'results' =>  $result,
            'link' =>  (string) $result->links()
        ];
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Product list successfully retrieved',
            'data'      => $this->data
        ], $this->code);
    }

    /**
     * 
     * @author  SOORYA
     */
    public function deleteItem(Request $request)
    {
        (new Item())->deleteItem($request->id);
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Product successfully deleted',
            'data'      => $this->data
        ], $this->code);
    }

    /**
     * 
     * 
     * 
     */
    public function list(Request $request)
    {
        $product = (new Item)->getItemById(base64_decode($request->id));
        return view('admin.products.list', ['product' => $product]);
    }

    public function getGalleryList(Request $request)
    {
        $result = (new itemImage())->getGalleryList($request);
        $this->data  = [
            'results' =>  $result,
            'link' =>  (string) $result->links()
        ];
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Gallery list successfully retrieved',
            'data'      => $this->data
        ], $this->code);
    }

    public function createAdminGalleryImage(Request $request)
    {
        $validation = Validator::make($request->all(), [
            'item_id'          => 'required',
            'name'             => 'required',
            'image'            => 'required'
        ]);
        if (!$validation->passes()) {
            return response()->json([
                'success'   => false,
                'message'   => $validation->errors()->first(),
                'data'      => $this->data
            ], $this->code);
        }

        $file =  $request->file('image');
        $extension = $file->extension();
        $file_name = implode('_', [
            date('YmdHis'),
            Str::slug($request->name, '_')
        ]);

        $path = 'images/gallery/';
        Storage::disk('local')->putFileAs('public/' . $path, $file, $file_name . '.' . $extension);

        (new itemImage())->createGallery([

            'item_id'               => $request->item_id,
            'name'                  => $request->name,
            'image'                 => '/storage/' . $path . $file_name . '.' . $extension,
        ]);

        return response()->json([
            'success'   => $this->success,
            'message'   => 'Gallery  successfully created',
            'data'      => $this->data
        ], $this->code);
    }

    public function getGalleryImage(Request $request)
    {
        $this->data  = ['image' => (new itemImage())->getGalleryImageById($request->id)];
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Gallery Image successfully retrieved',
            'data'      => $this->data
        ], $this->code);
    }

    public function updateGalleryImage(Request $request)
    {
        $validation = Validator::make($request->all(), [
            'name'                => 'required',
        ]);
        if (!$validation->passes()) {
            return response()->json([
                'success'   => false,
                'message'   => $validation->errors()->first(),
                'data'      => $this->data
            ], $this->code);
        }
        $data = [

            'name'                       => $request->name
        ];
        if ($request->image) {
            $file =  $request->file('image');
            $extension = $file->extension();
            $file_name = implode('_', [
                date('YmdHis'),
                Str::slug($request->name, '_')
            ]);

            $path = 'images/gallery/';

            Storage::disk('local')->putFileAs('public/' . $path, $file, $file_name . '.' . $extension);
            $data['image']  = '/storage/' . $path . $file_name . '.' . $extension;
        }
        (new itemImage())->updateGalleryImage($request->id, $data);
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Banner Image successfully updated',
            'data'      => $this->data
        ], $this->code);
    }

    public function deleteGalleryImage(Request $request)
    {
        (new itemImage())->galleryDelete($request->id);
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Gallery  successfully deleted',
            'data'      => $this->data
        ], $this->code);
    }

    /**
     * 
     * 
     */
    public function productDeliveryChargeListPage(Request $request)
    {
        $exclude_id             = [];
        $item                   = (new Item())->getItemById(base64_decode($request->id));
        $delivery_charge        = ProductDeliveryCharge::where('item_id', base64_decode($request->id))->get();
        $exclude_id             = $delivery_charge->pluck('state_id');
        $states                 = State::select('id', 'name')->whereNotIn('id', $exclude_id)->get();
        return view('admin.products.product-delivery-charge-list', ['item' => $item, 'states' => $states]);
    }

    /**
     * 
     * 
     */
    public function addProductDeliveryCharge(Request $request)
    {
        $rules               = $this->addDeliveryChargeRules();
        $this->validate($request, $rules);

        $delivery_charge     = ProductDeliveryCharge::where('state_id', $request->state_id)->where('item_id', $request->id)->first();
        if ($delivery_charge) {
            $this->success   = false;
            $this->message   = "Delivery charge Already Exits.Update it.";
        } else {
            $item            = (new Item())->getItemById($request->id);
            $State           = State::where('id', $request->state_id)->first();
            $delivery_charge = (new ProductDeliveryCharge())->createDeliveryCharge($request, $item, $State);
            if ($delivery_charge) {
                Item::where('id', $request->id)->update(['is_draft' => Carbon::now()->format('Y-m-d H:i:s')]);
            }
            $this->message   = "Delivery charge created successfully.";
        }
        // response
        $request->session()->flash('message', 'Delivery charge created successfully.');
        $request->session()->flash('alert-class', 'alert-success');
        return redirect('/admin/products/' . base64_encode($request->id) . '/delivery-charge');
    }


    /**
     * 
     * 
     */
    public function getProductDeliveryChargeList(Request $request)
    {
        $result         =  (new ProductDeliveryCharge())->getProductDeliveryChargeList($request);
        $this->data     =  [
            'results' =>  $result,
            'link' =>  (string) $result->links()
        ];
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Delivery charge successfully retrieved',
            'data'      => $this->data
        ], $this->code);
    }

    /**
     * 
     * 
     */
    public function getProductDeliveryCharge(Request $request)
    {
        $this->data         =  [
            'delivery' => (new ProductDeliveryCharge())->getProductDeliveryCharge($request->id)
        ];
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Delivery charge successfully retrieved',
            'data'      => $this->data
        ], $this->code);
    }

    /**
     * 
     * 
     */
    public function updateProductDeliveryCharge(Request $request)
    {
        $rules              = $this->updateProductDeliveryChargeRules();
        $this->validate($request, $rules);

        (new ProductDeliveryCharge())->updateProductDeliveryCharge($request);

        // response
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Delivery boy bank details successfully updated',
            'data'      => $this->data
        ], $this->code);
    }

    public function deleteProductDeliveryCharge(Request $request)
    {
        (new ProductDeliveryCharge())->deleteProductDeliveryCharge($request->id);
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Delivery Charge successfully deleted',
            'data'      => $this->data
        ], $this->code);
    }

    /**
     * 
     * Review
     * 
     */
    public function reviewItem(Request $request)
    {
        $item           = (new Item())->getItemById(base64_decode($request->id));
        $total_review   =  Review::where('item_id', base64_decode($request->id))->count();
        $rating         =  Review::where('item_id', base64_decode($request->id))->avg('rating');
        $total_rating   =  sprintf("%0.1f", $rating);
        return view('admin.products.review-list', [
            'item'              => $item,
            'total_review'      => $total_review,
            'total_rating'      => $total_rating
        ]);
    }

    public function getReviewList(Request $request)
    {

        $result         =  (new Review())->getReviewList($request);
        $this->data     =  [
            'results' =>  $result,
            'link'    =>  (string) $result->links()
        ];
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Review successfully retrieved',
            'data'      => $this->data
        ], $this->code);
    }

    public function delete(Request $request)
    {

        (new Review())->deleteReview($request->id);
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Review successfully deleted',
            'data'      => $this->data
        ], $this->code);
    }

    public function editReview(Request $request)
    {
        $review = (new Review())->getReviewById(base64_decode($request->id));
        return view('admin.products.review-edit', ['review' => $review]);
    }

    public function updateReview(Request $request)
    {
        $rules              = $this->updateReviewRules(base64_decode($request->id));
        $this->validate($request, $rules);

        $review             = (new Review())->getReviewById(base64_decode($request->id));
        if (!$review) {
            $this->success  = false;
            $this->message  = "Failed to update review.";
        } else {
            $update_review =  (new Review())->updateReview($request, $review->id);
            $this->message = 'review successfully updated';
        }

        $request->session()->flash('message', 'Review successfully updated');
        $request->session()->flash('alert-class', 'alert-success');
        return redirect('admin/products/' . base64_encode($review->item_id) . '/review');
    }

    /**
     * 
     *  @author SOORYA
     */
    public function getDraftProductList(Request $request)
    {
        $result         =  (new Item())->getDraftProductList($request);
        $this->data     =  [
            'results' =>  $result,
            'link' =>  (string) $result->links()
        ];
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Draft Products successfully retrieved',
            'data'      => $this->data
        ], $this->code);
    }

    /**
     * 
     *  @author SOORYA
     */
    public function getUnverifiedProductList(Request $request)
    {
        $result         =  (new Item())->getUnverifiedProductList($request);
        $this->data     =  [
            'results' =>  $result,
            'link' =>  (string) $result->links()
        ];
        return response()->json([
            'success'   => $this->success,
            'message'   => 'Unverified Products successfully retrieved',
            'data'      => $this->data
        ], $this->code);
    }
    /**
     * 
     * @author SOORYA
     */
    public function viewProduct(Request $request)
    {
        $product     = (new Item)->getItemByIdVerified(base64_decode($request->id));
        $this->data  =  [
            'product'          =>  $product,
            'categories'       => (new Category())->getCategories(),
            'vendors'          => (new Vendor())->getVendors(),
            'product_types'    => (new productType())->getProductTypes()
        ];
        return view('admin.products.view-product', $this->data);
    }

    /**
     * 
     * 
     */
    public function getItemVariants(Request $request)
    {
        $item             = (new Item())->getItemById(base64_decode($request->id));
        $variants         =  Item::where('parent_id', base64_decode($request->id))->get();
        return view('admin.products.product-vriants-list', [
            'item'              => $item,
            'variants'          => $variants
        ]);
    }

    /**
     * 
     * @author SOORYA
     */
    public function updateUnverifiedProductApproveStatus(Request $request)
    {
        $item                        = (new Item())->getItemById($request->id);
        if ($item) {
            $change_approve_status       = (new Item())->UnverifiedChangeApproveStatus($request->id);
            $title              = "Verified: " . $item->name;
            $image              =  config('app.url') . $item->image;
            $message            =  "The Product " . $item->name . " Created at " . $item->created_at . " is verified";
            $firebase_token     =  (new UserDevice())->userFirebaseTokenList($item->vendor->user->id);
            $type               = "Poduct-Verification";
            ProductVerificationJob::dispatch($title, $image, $message, $firebase_token, $type);
            // response
            $request->session()->flash('message', 'Product verified successfully.');
            $request->session()->flash('alert-class', 'alert-success');
            return redirect('/admin/unverified-product');
        } else {
            // response
            $request->session()->flash('message', 'Product doesnot exis.');
            $request->session()->flash('alert-class', 'alert-danger');
            return redirect('/admin/unverified-product');
        }
    }

    public function addDeliveryChargeRules()
    {
        $rules = [
            'state_id'          => 'required',
            'delivery_charge'   => 'required|numeric'
        ];
        return $rules;
    }

    public function updateProductDeliveryChargeRules()
    {
        $rules = [
            'state_id'          => 'required',
            'delivery_charge'   => 'required|numeric'
        ];
        return $rules;
    }

    public function updateReviewRules($id)
    {
        $rules = [
            'rating'              => 'required|numeric',
            'review'              => 'required'
        ];
        return $rules;
    }
}
