<?php

namespace App\Http\Controllers;

use App\Models\Banner;
use App\Models\Category;
use App\Models\Item;
use App\Models\productType;
use App\Models\State;
use App\Models\User;
use App\Models\UserAddress;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class FrontendController extends Controller
{
    /**
     * Http response data
     */
    private $data;

    /**
     * Http response code
     */
    private $code;

    /**
     * Response status
     */
    private $success;

    /**
     * Response message
     */
    private $message;

    /**
     * Init class attributes
     */
    public function __construct(Request $request)
    {
        $this->data     = new \stdClass();
        $this->code     = Response::HTTP_OK;
        $this->success  = true;
        $this->message  = '';
    }
    public function getProductList(Request $request)
    {
        $keyword        = $request->keyword;
        $categorySlug   = $request->category;
        $tagsSlug       = $request->tag;

        $products = Item::search($keyword)
            ->when($categorySlug, function ($query) use ($categorySlug) {
                $query->where('categories', $categorySlug);
            })
            ->when($tagsSlug, function ($query) use ($tagsSlug) {
                $query->where('tags', $tagsSlug);
            })
            ->paginate(20);
        $this->data = ['products' => $products];
        return response()->json([
            'success'   => $this->success,
            'message'   => $this->message,
            'data'      => $this->data
        ], $this->code);
    }

    public function index()
    {
        $banners                = (new Banner())->getBannerList();
        $featured_products      = (new Item())->getFeaturedProducts(1, 12);
        $new_arrivals           = (new Item())->getFeaturedProducts(2, 7);
        $footer_bestsellings    = (new Item())->getFeaturedProducts(3, 4);
        $footer_featureds       = (new Item())->getFeaturedProducts(1, 4);
        $footer_newarrivals     = (new Item())->getFeaturedProducts(2, 4);
        $footer_rated           = (new Item())->getFeaturedProducts(4, 4);
        return view('frontend.index', compact('banners', 'featured_products', 'new_arrivals', 'footer_bestsellings', 'footer_featureds', 'footer_newarrivals', 'footer_rated'));
    }

    public function products()
    {
        $categories = (new Category())->getFiltterCategory();
        $tags       = (new productType())->getFiltterTags();
        return view('frontend.products', compact('categories', 'tags'));
    }

    public function getCategoryProductList($slug)
    {
        $categoryData     = (new Category())->getCategoryWithSubcategories($slug);
        return view('frontend.category-products', compact('categoryData'));
    }

    public function getTagsProductList($slug)
    {

        $tag     = (new productType())->getProductTypeBySlug($slug);
        return view('frontend.tags-products', compact('tag'));
    }

    public function productDetails($slug)
    {
        $item               = (new Item())->getItemWithDetails($slug);
        $similar_products   = (new Item())->getSimilarProducts($item->id);
        $variants           = (new Item())->getVariants($item->id);
        return view('frontend.product_details', compact('item', 'similar_products', 'variants'));
    }

    public function about()
    {
        return view('frontend.about');
    }

    public function contact()
    {
        return view('frontend.contact');
    }

    public function blogs()
    {
        return view('frontend.blog');
    }

    public function blogDetails()
    {
        return view('frontend.blog_details');
    }

    public function getSearch(Request $request)
    {
        $query = $request->query('query');
        $search = Item::search($query)->get();
        $this->data = ['search' => $search];

        return response()->json([
            'success'   => $this->success,
            'message'   => $this->message,
            'data'      => $this->data
        ], $this->code);
    }

    public function getCartList(Request $request)
    {

        return view('frontend.cart');
    }

    public function getCheckout(Request $request)
    {
        $user_id = Auth::user()->id;
        $user_address   = (new UserAddress())->getDefaultAddress($user_id);
        $states         = State::all();
        return view('frontend.checkout', compact('states', 'user_address'));
    }


    public function getCheckoutPayment(Request $request)
    {
        $user_id = Auth::user()->id;
        $user_address   = (new UserAddress())->getDefaultAddress($user_id);
        $states         = State::all();
        return view('frontend.checkout-payment', compact('states', 'user_address'));
    }

    public function getUserLogin()
    {
        return view('frontend.user-login');
    }

    public function getUserRegister()
    {
        return view('frontend.user-register');
    }

    public function setUserRegister(Request $request)
    {

        $validatedData = $request->validate([
            'name'      => 'required|string|max:255',
            'phone'     => 'required|string|min:10|max:10|unique:users,phone',
            'email'     => 'required|string|email|max:255|unique:users,email',
            'password'  => 'required|string|min:6|confirmed',
        ]);


        $user = User::create([
            'name'      => $validatedData['name'],
            'phone'     => $validatedData['phone'],
            'email'     => $validatedData['email'],
            'username'  => $validatedData['email'],
            'password'  => Hash::make($validatedData['password']),
            'role_id'   => 1,
        ]);
        return redirect()->route('user-login')->with('success', 'Registration successful!');
    }

    public function setUserLogin(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required|string',
        ]);

        $credentials = $request->only('email', 'password');
        if (Auth::attempt($credentials, $request->filled('remember'))) {
            return redirect()->intended('user-dashboard')->with('success', 'You are logged in!');
        }
        return back()->withErrors([
            'email' => 'The provided credentials do not match.',
        ])->withInput($request->except('password'));
    }


    public function userLogout(Request $request)
    {
        Auth::logout();
        return redirect('/user-login')->with('success', 'Logged out successfully!');
    }


    public function getUserDashBoard()
    {
        return view('frontend.user-dashboard');
    }

    public function saveUserAddress(Request $request)
    {

        $validatedData = $request->validate([
            'contact_name'      => 'required|string|max:255',
            'email'             => 'required|email|max:255',
            'mobile_number'     => 'required|string|max:15',
            'pincode'           => 'required|string|max:10',
            'state_id'          => 'required|exists:states,id',
            'house_name'        => 'required|string|max:255',
            'landmark'          => 'nullable|string|max:255',
            'address_id'        => 'nullable|exists:user_addresses,id',
        ]);

        $userId = Auth::id();
        $validatedData['is_primary'] = 1;
        if ($request->address_id) {
            $address = UserAddress::findOrFail($request->address_id);
            $address->update($validatedData);
            $message = 'Address updated successfully!';
        } else {

            $validatedData['user_id'] = $userId;
            UserAddress::create($validatedData);
            $message = 'Address saved successfully!';
        }
        return redirect()->route('user.checkout-payment')->with('success',  $message);
    }
}
