<?php

namespace App\Http\Controllers;

use App\Helpers\CommonHelper;
use App\Models\Authors;
use App\Models\Category;
use Illuminate\Support\Facades\Storage;
use App\Models\GoogleSheet;
use App\Models\productType;
use App\Services\GoogleSheetsService;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Validator;

class GoogleSheetController extends Controller
{

    /**
     * Http response data
     */
    private $data;

    /**
     * Http response code
     */
    private $code;

    /**
     * Response status
     */
    private $success;

    /**
     * Response message
     */
    private $message;

    /**
     * Init class attributes
     */


    protected $googleSheetsService;
    public function __construct(GoogleSheetsService $googleSheetsService)
    {
        $this->googleSheetsService = $googleSheetsService;
        $this->data     = new \stdClass();
        $this->code     = Response::HTTP_OK;
        $this->success  = true;
        $this->message  = '';
    }


    public function getFetchSheetData(Request $request)
    {
        GoogleSheet::truncate();
        $range = 'Sheet1';
        $data = $this->googleSheetsService->getSheetData($range);
        array_shift($data);
        foreach ($data as $row) {
            GoogleSheet::create([
                'sku' => $row[0],
                'title' => $row[1],
                'description' => $row[2] ?? null,
                'mrp' => $row[3],
                'price' => $row[4],
                'image' => $row[5] ?? null,
                'gallery1' => $row[6] ?? null,
                'gallery2' => $row[7] ?? null,
                'gallery3' => $row[8] ?? null,
                'gallery4' => $row[9] ?? null,
                'feature1' => $row[10] ?? null,
                'feature2' => $row[11] ?? null,
                'feature3' => $row[12] ?? null,
                'feature4' => $row[13] ?? null,
                'feature5' => $row[14] ?? null,
                'product_specification' => $row[15] ?? null,
                'author'        => $row[16] ?? null,
                'category'      => $row[17] ?? null,
                'tags'          => $row[18] ?? null,
                'language'      => $row[19] ?? null,
                'stock'         => $row[20] ?? null,
                'is_completed'  => $row[21] ?? null,
            ]);
        }
        return response()->json([
            'success'   => $this->success,
            'message'   => 'product list from sheet fetched successfully',
            'data'      => $this->data
        ], $this->code);
    }


    public function getCategoriesForSheet()
    {
        $this->data = ["categories" => (new Category())->all('id', 'name')];
        return response()->json([
            'success'   => $this->success,
            'message'   => 'category list fetched successfully',
            'data'      => $this->data
        ], $this->code);
    }

    public function getTagsForSheet()
    {
        $this->data = ["tags" => (new productType())->all('id', 'name')];
        return response()->json([
            'success'   => $this->success,
            'message'   => 'tags list fetched successfully',
            'data'      => $this->data
        ], $this->code);
    }



    public function getAuthorForSheet()
    {
        $this->data = ["authors" => (new Authors())->all('id', 'name')];
        return response()->json([
            'success'   => $this->success,
            'message'   => 'authors list fetched successfully',
            'data'      => $this->data
        ], $this->code);
    }



    public function getSheetData(Request $request)
    {
        $search = $request->query('search');
        $query = GoogleSheet::query();
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('id', 'like', "%{$search}%")
                    ->orWhere('SKU', 'like', "%{$search}%")
                    ->orWhere('title', 'like', "%{$search}%");
            });
        }
        $paginatedData = $query->paginate(50);
        $this->data = [
            "products"     => $paginatedData->items(),
            'current_page' => $paginatedData->currentPage(),
            'total_pages'  => $paginatedData->lastPage(),
            'total_items'  => $paginatedData->total(),
            'per_page'     => $paginatedData->perPage(),
        ];
        return response()->json([
            'success'   => $this->success,
            'message'   => 'product list fetched successfully',
            'data'      => $this->data
        ], $this->code);
    }

    public function updateSheetData(Request $request)
    {
        $product = GoogleSheet::find($request->id);
        if (!$product) {
            $this->success = false;
            $this->message = 'Product not found';
        } else {
            $validator = Validator::make(
                $request->all(),
                $this->dataUpdateRule(),
                (new CommonHelper())->getValidationMessage()
            );
            if ($validator->fails()) {
                $this->success                = false;
                $this->message                = $validator->errors()->first();
            } else {
                $data = [
                    'title' => $request->title ?? $product->title,
                    'description' => $request->description ?? $product->description,
                    'mrp' => $request->mrp ?? $product->mrp,
                    'price' => $request->price ?? $product->price,
                    'feature1' => $request->feature1 ?? $product->feature1,
                    'feature2' => $request->feature2 ?? $product->feature2,
                    'feature3' => $request->feature3 ?? $product->feature3,
                    'feature4' => $request->feature4 ?? $product->feature4,
                    'feature5' => $request->feature5 ?? $product->feature5,
                    'product_specification' => $request->product_specification ?? $product->product_specification,

                    'author'        => $request->author ?? $product->author,
                    'category'      => $request->category ?? $product->category,
                    'tags'          => $request->tags ?? $product->tags,
                    'language'      => $request->language ?? $product->language,
                    'stock'         => $request->stock ?? $product->stock,
                    'is_completed'  => $request->is_completed ?? $product->is_completed,
                ];
                $data['image'] = $this->updateAndStoreImage($request->image ?? null, $product->image);
                $data['gallery1'] = $this->updateAndStoreImage($request->gallery1 ?? null, $product->gallery1);
                $data['gallery2'] = $this->updateAndStoreImage($request->gallery2 ?? null, $product->gallery2);
                $data['gallery3'] = $this->updateAndStoreImage($request->gallery3 ?? null, $product->gallery3);
                $data['gallery4'] = $this->updateAndStoreImage($request->gallery4 ?? null, $product->gallery4);
                $product->update($data);
                $this->success = true;
                $this->message = 'successfully update';
            }
        }
        return response()->json([
            'success'   => $this->success,
            'message'   => $this->message,
            'data'      => $this->data
        ], $this->code);
    }

    private function updateAndStoreImage($newImage, $existingImagePath)
    {
        if ($newImage) {
            if ($existingImagePath && Storage::exists($existingImagePath)) {
                Storage::delete($existingImagePath);
            }
            $image_url =  "https://bookatdoor.com/storage/" . $newImage->store('sheet_image', 'public');
            return $image_url;
        }
        return $existingImagePath;
    }
    public function dataUpdateRule()
    {
        return [
            'title' => 'nullable|string',
            'description' => 'nullable|string',
            'mrp' => 'nullable|numeric',
            'price' => 'nullable|numeric',
            'image' => 'nullable|file|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'gallery1' => 'nullable|file|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'gallery2' => 'nullable|file|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'gallery3' => 'nullable|file|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'gallery4' => 'nullable|file|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'feature1' => 'nullable|string',
            'feature2' => 'nullable|string',
            'feature3' => 'nullable|string',
            'feature4' => 'nullable|string',
            'feature5' => 'nullable|string',
            'product_specification' => 'nullable|string',
        ];
    }


    public function AddDataSheetDataSingleRow(Request $request)
    {
        $sheet_row = $request->id + 1;
        $id = $request->id;
        $range = "Sheet1!A{$sheet_row }";
        $values = GoogleSheet::where('id', $id)
            ->select(
                'sku',
                'title',
                'description',
                'mrp',
                'price',
                'image',
                'gallery1',
                'gallery2',
                'gallery3',
                'gallery4',
                'feature1',
                'feature2',
                'feature3',
                'feature4',
                'feature5',
                'product_specification',
                'author',
                'category',
                'tags',
                'language',
                'stock',
            )
            ->get()
            ->map(function ($item) {
                return [
                    $item->sku ?? null,
                    $item->title ?? null,
                    $item->description ?? null,
                    $item->mrp ?? null,
                    $item->price ?? null,
                    $item->image ?? null,
                    $item->gallery1 ?? null,
                    $item->gallery2 ?? null,
                    $item->gallery3 ?? null,
                    $item->gallery4 ?? null,
                    $item->feature1 ?? null,
                    $item->feature2 ?? null,
                    $item->feature3 ?? null,
                    $item->feature4 ?? null,
                    $item->feature5 ?? null,
                    $item->product_specification ?? null,
                    $item->author ?? null,
                    $item->category ?? null,
                    $item->tags ?? null,
                    $item->language ?? null,
                    $item->stock ?? null
                ];
            })
            ->toArray();

        $updateResponse = $this->googleSheetsService->updateSheetData($range, $values);
        return response()->json($updateResponse);
    }

    public function AddDataSheetData()
    {

        $range = 'Sheet1!A2';
        $values = GoogleSheet::select(
            'sku',
            'title',
            'description',
            'mrp',
            'price',
            'image',
            'gallery1',
            'gallery2',
            'gallery3',
            'gallery4',
            'feature1',
            'feature2',
            'feature3',
            'feature4',
            'feature5',
            'product_specification',
            'author',
            'category',
            'tags',
            'language',
            'stock',
        )
            ->get()
            ->map(function ($item) {
                return [
                    $item->sku ?? null,
                    $item->title ?? null,
                    $item->description ?? null,
                    $item->mrp ?? null,
                    $item->price ?? null,
                    $item->image ?? null,
                    $item->gallery1 ?? null,
                    $item->gallery2 ?? null,
                    $item->gallery3 ?? null,
                    $item->gallery4 ?? null,
                    $item->feature1 ?? null,
                    $item->feature2 ?? null,
                    $item->feature3 ?? null,
                    $item->feature4 ?? null,
                    $item->feature5 ?? null,
                    $item->product_specification ?? null,
                    $item->author ?? null,
                    $item->category ?? null,
                    $item->tags ?? null,
                    $item->language ?? null,
                    $item->stock ?? null
                ];
            })
            ->toArray();

        $updateResponse = $this->googleSheetsService->updateSheetData($range, $values);
        return response()->json($updateResponse);
    }

    public function updateItemCategory(Request $request)
    {
        $item_id      = $request->input('id');
        $categoryIds    = $request->input('categories');

        $item           = GoogleSheet::find($item_id);
        if ($item) {

            $item->category = implode(",", $categoryIds);
            $item->save();
            $this->success = true;
            $this->message = "Success fully update";
        } else {
            $this->success = false;
            $this->message = "something went wrong";
        }
        return response()->json([
            'success'   => $this->success,
            'message'   => $this->message,
            'data'      => $this->data
        ], $this->code);
    }

    public function updateItemTags(Request $request)
    {
        $item_id      = $request->input('id');
        $tags    = $request->input('tags');

        $item           = GoogleSheet::find($item_id);
        if ($item) {

            $item->tags = implode(",", $tags);
            $item->save();
            $this->success = true;
            $this->message = "Success fully update";
        } else {
            $this->success = false;
            $this->message = "something went wrong";
        }
        return response()->json([
            'success'   => $this->success,
            'message'   => $this->message,
            'data'      => $this->data
        ], $this->code);
    }

    public function updateItemDescription(Request $request)
    {
        $item_id      = $request->input('id');
        $description  = $request->input('description');

        $item           = GoogleSheet::find($item_id);
        if ($item) {

            $item->description = $description;
            $item->save();
            $this->success = true;
            $this->message = "Success fully update";
        } else {
            $this->success = false;
            $this->message = "something went wrong";
        }
        return response()->json([
            'success'   => $this->success,
            'message'   => $this->message,
            'data'      => $this->data
        ], $this->code);
    }

    public function updateAuthor(Request $request)
    {
        $item_id      = $request->input('id');
        $author       = $request->input('author');
        $item           = GoogleSheet::find($item_id);
        if ($item) {
            $item->author = $author;
            $item->save();
            $this->success = true;
            $this->message = "Success fully update";
        } else {
            $this->success = false;
            $this->message = "something went wrong";
        }
        return response()->json([
            'success'   => $this->success,
            'message'   => $this->message,
            'data'      => $this->data
        ], $this->code);
    }
}
