<?php

namespace App\Models;

use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use Auth;

class Category extends Model
{
    use SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'parent_id',
        'name',
        'slug',
        'logo',
        'banner',
        'is_active',
        'is_home'
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'created_at',
        'updated_at',
        'deleted_at',
        'pivot'
    ];

    public static function boot()
    {
        parent::boot();
        static::saving(function ($category) {
            if (empty($category->slug)) {
                $category->slug = Str::slug($category->name);
            }
        });
    }

    /**
     * 
     * 
     * 
     */
    public function products()
    {
        return $this->hasMany('App\Models\Item');
    }

    public function childCategory()
    {
        return $this->hasMany(\App\Models\Category::class,  'parent_id','id');
    }

    public function parent()
    {
        return $this->belongsTo(\App\Models\Category::class,'parent_id','id');
    }
    /**
     *Pivot Table
     *@author SOORYA
     */
    public function items()
    {
        return $this->belongsToMany(\App\Models\Item::class)->with('gallery')->withTimestamps();
    }

    /**
     * 
     * @return void
     */
    public function getImageAttribute($value)
    {
        return url("/") . $value;
    }

    /**
     *
     *@author RAHUL
     */
    public function getMainCategory()
    {
        return self::select('id', 'name')->whereNull('parent_id')->get();
    }

    /**
     *
     *@author RAHUL
     */
    public function getAllCategories()
    {
        return self::select('id', 'name')->get();
    }

    /**
     *
     *@author RAHUL
     */
    public function getAdminCategories()
    {
        return self::get();
    }


    /**
     *
     *
     *@author RAHUL
     */
    public function getSingleCategory($id)
    {
        return self::where('id', $id)->first();
    }

    public function getCategoryById($id)
    {
        return self::find($id);
    }

    /**
     *
     *
     *@author RAHUL
     */
    public function deleteCategory($id)
    {
        return self::where('id', $id)->delete();
    }


    /**   
     *
     *@author RAHUL
     */
    public function updateCategory($id, $input)
    {
        return self::where('id', $id)->update($input);
    }

    /**   
     *
     *@author RAHUL
     */
    public function createCategory($input)
    {
        return self::create($input);
    }


    public function getCategories()
    {
        return self::whereNull('parent_id')->get();
    }

    /**
     *Pivot Table
     *@author SOORYA
     */
    public function getExcludedCategories($selected_categories)
    {
        $data = [];
        for ($i = 0; $i < count($selected_categories); $i++) {
            $data[] = $selected_categories[$i]->category_id;
        }

        return self::whereNotIn('id', $data)->get();
    }

    /**
     *
     *@author RAHUL
     */
    public function getAdminCategoryList($request)
    {
        return self::select('id', 'is_active', 'name', 'is_home', 'logo', 'banner', 'slug', 'parent_id')->with('parent')->paginate($request->limit);
    }



    public function getCategoryWithProducts($categories)
    {
        return self::select('id', 'name')
            ->whereIn('id', $categories)
            ->get();
    }

    public function changeStatus($id)
    {
        $category =  self::where('id', $id)->first();
        if ($category->is_home == 1) {
            return self::where('id', $id)->update(['is_home' => 0]);
        } else {
            return self::where('id', $id)->update(['is_home' => 1]);
        }
    }

    public function getUserCategory()
    {
        return self::select('id', 'name', 'logo')
            ->where('is_home', 1)
            ->get();
    }

    public function getCategoryProductsByVendorId($vendor_id)
    {
        return self::select('id', 'name', 'is_active')
            ->whereHas('items', function ($query) use ($vendor_id) {
                $query->where('vendor_id', $vendor_id)
                    ->where('is_verified', '!=', null);
            })
            ->with(['items' => function ($query) use ($vendor_id) {
                $query->with('item_image', 'categories:id,name', 'product_types:id,name')
                    ->where('vendor_id', $vendor_id)
                    ->where('is_verified', '!=', null);
            }])
            ->orderBy('id', 'ASC')
            ->groupBy('id')
            ->get();
    }


    public function getFiltterCategory()
    {
        return self::all();
    }

    public function getCategoryWithSubcategories($slug)
    {
        return self::where('slug', $slug)->with('childCategory')->first();
    }
}
