<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Carbon;
use Laravel\Scout\Searchable;
use Auth;
use App\Helpers\CommonHelper;
use DB;


class Item extends Model
{
    use Searchable;

    // use \Staudenmeir\EloquentJsonRelations\HasJsonRelationships;
    use SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'uid',
        'sku',
        'slug',
        'name',
        'descriptions',
        'specification',
        'features',
        'image',
        'mrp',
        'price',
        'vendor_id',
        'parent_id',
        'color',
        'size',
        'stock',
        'is_offer',
        'is_verified',
        'is_draft',

    ];
    public function searchableAs(): string
    {
        return 'items_index';
    }

    public function toSearchableArray(): array
    {
        $this->loadMissing('categories');
        $this->loadMissing('product_type');
        $categorySlugs = $this->categories->pluck('slug')->toArray();
        $tagsSlug = $this->product_types->pluck('slug')->toArray();
        return [
            'id'            => (int) $this->id,
            'slug'          => $this->slug,
            'name'          => $this->name,
            'descriptions'  => $this->descriptions,
            'image'         => $this->image,
            'price'         => (float) $this->price,
            'mrp'           => (float) $this->mrp,
            'color'         => (float) $this->color,
            'size'          => (float) $this->size,
            'parent_id'     => (float) $this->parent_id,
            'categories'    => $categorySlugs,
            'tags'          => $tagsSlug,

        ];
    }
    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'category_id'     => 'array',
        'product_type_id' => 'array'
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'updated_at',
        'deleted_at',
        'pivot'
    ];

    public function getStockAttribute($value)
    {
        return boolval($value);
    }

    /**
     * @author PMS
     */
    public function getDashboardProductListTotalPage($limit)
    {
        $result    =  self::select('name', 'image', 'price')
            ->where('is_verified', '!=', null)
            ->orderBy('id', 'desc')
            ->with('vendor')
            ->whereHas('vendor', function ($q) {
                $q->where('is_active', 1);
            })
            ->paginate(($limit) ? $limit : 12);
        $total_page = (int)ceil($result->total() / $result->perPage());
        return $total_page;
    }

    /**
     * @author PMS
     */
    public function getDashboardProductLis($limit, $offset, $reseller_id)
    {
        return   self::select('id', 'name', 'image', 'price')
            ->where('is_verified', '!=', null)
            ->orderBy('id', 'desc')
            ->offset(($offset) ? $offset : 0)
            ->limit(($limit) ? $limit : 12)
            ->withCount(['reviews as rating' => function ($query) {
                $query->select(DB::raw('round(avg(rating),1)'));
            }])
            ->with(['wishlist' => function ($q) use ($reseller_id) {
                $q->where('reseller_id', $reseller_id);
            }])
            ->with('vendor', 'gallery')
            ->whereHas('vendor', function ($q) {
                $q->where('is_active', 1);
            })
            ->get();
    }
    /**
     *
     *@author SOORYA
     */
    public static function boot()
    {
        parent::boot();
        self::creating(function ($model) {
            $model->uid = (string) Str::uuid();
        });
    }



    /**
     *
     *@author SOORYA
     */
    public function category()
    {
        return $this->belongsTo(\App\Models\Category::class);
    }

    public function vendor()
    {
        return $this->belongsTo(\App\Models\Vendor::class);
    }

    public function product_type()
    {
        return $this->belongsTo(\App\Models\productType::class);
    }
    /**
     * Pivot Table
     *@author SOORYA
     */
    public function product_types()
    {
        return $this->belongsToMany(\App\Models\productType::class)->withTimestamps();
    }
    /**
     *Pivot Table
     *@author SOORYA
     */
    public function categories()
    {
        return $this->belongsToMany(Category::class, 'category_item')->withTimestamps();
    }

    public function reviews()
    {
        return $this->hasMany('App\Models\Review');
    }

    public function wishlist()
    {
        return $this->hasMany('App\Models\Wishlist');
    }

    public function product_delivery_charge()
    {
        return $this->hasMany('App\Models\ProductDeliveryCharge');
    }

    public function item_image()
    {
        return $this->hasMany('App\Models\itemImage');
    }

    public function gallery()
    {
        return $this->hasMany('App\Models\itemImage')->select('id', 'name', 'image', 'item_id');
    }


    /**
     * @author PMS
     */
    public function getNewArrivals($reseller_id)
    {
        return self::select(
            'id',
            'uid',
            'name',
            'descriptions',
            'image',
            'price',
            'vendor_id'
        )
            ->withCount(['reviews as rating' => function ($query) {
                $query->select(DB::raw('round(avg(rating),1)'));
            }])
            ->with(['wishlist' => function ($q) use ($reseller_id) {
                $q->where('reseller_id', $reseller_id);
            }])
            ->with('vendor', 'gallery')
            ->whereHas('vendor', function ($q) {
                $q->where('is_active', 1);
            })
            ->where('is_verified', '!=', null)
            ->orderBy('id', 'DESC')
            ->limit(5)
            ->get();
    }

    /**
     * 
     */
    public function getRating($item_id)
    {
        return self::withCount(['reviews as rating' => function ($query) {
            $query->select(DB::raw('round(avg(rating),1)'));
        }])
            ->withCount(['reviews as review_count' => function ($query) {
                $query->select(DB::raw('count(review)'));
            }])
            ->where('id', $item_id)
            ->first();
    }

    /**
     * @author PMS
     */
    public function getTrendy($reseller_id)
    {
        return self::select(
            'id',
            'uid',
            'name',
            'descriptions',
            'image',
            'price',
            'vendor_id'
        )
            ->withCount(['reviews as rating' => function ($query) {
                $query->select(DB::raw('round(avg(rating),1)'));
            }])
            ->with(['wishlist' => function ($q) use ($reseller_id) {
                $q->where('reseller_id', $reseller_id);
            }])
            ->with('vendor', 'gallery')
            ->whereHas('vendor', function ($q) {
                $q->where('is_active', 1);
            })
            ->where('is_verified', '!=', null)
            ->limit(5)
            ->get();
    }

    /**
     * @author PMS
     */
    public function getOfferItems($reseller_id)
    {
        return self::select(
            'id',
            'uid',
            'name',
            'descriptions',
            'image',
            'price',
            'vendor_id'
        )
            ->withCount(['reviews as rating' => function ($query) {
                $query->select(DB::raw('round(avg(rating),1)'));
            }])
            ->with(['wishlist' => function ($q) use ($reseller_id) {
                $q->where('reseller_id', $reseller_id);
            }])
            ->with('vendor', 'gallery')
            ->whereHas('vendor', function ($q) {
                $q->where('is_active', 1);
            })
            ->where('is_verified', '!=', null)
            ->where('is_offer', 1)
            ->limit(5)
            ->get();
    }
    /**
     *@author SOORYA
     */
    public function orders()
    {
        return $this->hasMany(\App\Models\OrderDetail::class);
    }

    /**
     * 
     */
    public function getIsMostSellingAttribute($value)
    {
        return $value ? boolval($value) : false;
    }


    /**
     *
     *@author SOORYA
     */
    public function createItem($input)
    {
        return self::create($input);
    }


    /**
     *
     *@author SOORYA
     */
    public function updateItem($id, $input)
    {
        return self::where('id', $id)->update($input);
    }

    public function getAdminTotalItems()
    {
        return self::count();
    }


    /**
     *
     *@author SOORYA
     */
    public function getItemById($id)
    {
        return self::find($id);
    }

    /**
     *
     *@author SOORYA
     */
    public function getItemByIdVerified($id)
    {
        return self::with('product_delivery_charge', 'item_image', 'vendor')
            ->where('id', $id)
            ->first();
    }

    /**
     *
     *@author SOORYA
     */
    public function getItemByUid($uid)
    {
        return self::where('uid', $uid)->first();
    }

    /**
     *          
     */

    public function deleteItem($id)
    {
        return self::where('id', $id)->delete();
    }

    /**
     *          
     */
    public function delVendorItems($id)
    {
        return self::where('vendor_id', $id)->delete();
    }
    /**
     *
     *@author SOORYA
     */
    public function getVendorItemId($vendor_id, $product_id)
    {
        return self::where('vendor_id', $vendor_id)
            ->where('id', $product_id)
            ->first();
    }
    /**
     *
     *@author SOORYA
     */
    public function updateStockStatus($request, $stock)
    {
        $data = [
            'stock' => $stock
        ];
        return self::where('id', $request->product_id)->update($data);
    }
    /**
     *
     *@author SOORYA
     */
    public function getAdminItemList($request)
    {
        $query =  self::where('is_draft', '!=', null)->whereNull('parent_id')->where('is_verified', '!=', null)->with('vendor');
        if (isset($request->category_id) && isset($request->category_id)) {
            $query->where('category_id', $request->category_id);
        }

        if (isset($request->search) && isset($request->search)) {
            $query->where('name', 'LIKE', '%' . $request->search . '%');
        }
        if (isset($request->sort_by) && isset($request->sort_by)) {
            $query->orderBy($request->sort_by);
        }
        return   $query->latest()->paginate($request->limit);
    }

    /**
     * 
     * 
     */

    public function getProductListTotalPage($category_id, $limit)
    {
        $result    =  self::join('category_item', 'items.id', '=', 'category_item.item_id')
            ->select('items.name', 'items.image', 'items.price')
            ->where('items.is_verified', '!=', null)
            ->where('category_item.category_id', $category_id)
            ->orderBy('items.id', 'desc')
            ->paginate(($limit) ? $limit : 12);
        $total_page = (int)ceil($result->total() / $result->perPage());
        return $total_page;
    }

    /**
     * 
     * 
     * 
     */
    public function getProductListHistory($category_id, $limit, $offset, $reseller_id)
    {
        return self::join('category_item', 'items.id', '=', 'category_item.item_id')
            ->join('vendors', 'items.vendor_id', '=', 'vendors.id')
            ->select('items.id', 'items.name', 'items.image', 'items.price')
            ->where('items.is_verified', '!=', null)
            ->where('vendors.is_active', 1)
            ->where('category_item.category_id', $category_id)
            ->orderBy('items.id', 'desc')
            ->withCount(['reviews as rating' => function ($query) {
                $query->select(DB::raw('round(avg(rating),1)'));
            }])
            ->with(['wishlist' => function ($q) use ($reseller_id) {
                $q->where('reseller_id', $reseller_id);
            }])
            ->with('gallery')
            ->groupBy('items.id')
            ->offset(($offset) ? $offset : 0)
            ->limit(($limit) ? $limit : 12)
            ->paginate($limit);
    }

    /**
     * 
     * @author PMS
     * 
     */
    public function getSearchProducts($search_key)
    {
        return   self::select('id', 'name', 'image', 'price')
            ->where('name', 'LIKE', '%' . $search_key . '%')
            ->with('vendor', 'gallery')
            ->whereHas('vendor', function ($q) {
                $q->where('is_active', 1);
            })
            ->limit(20)
            ->get();
    }

    /**
     * 
     * 
     */
    public function getProductListData($request)
    {
        $query  = self::join('category_item', 'items.id', '=', 'category_item.item_id')
            ->join('vendors', 'items.vendor_id', '=', 'vendors.id')
            ->select('items.id', 'items.name', 'items.image', 'items.price')
            ->where('items.is_verified', '!=', null)
            ->where('vendors.is_active', 1)
            ->where('category_item.category_id', $request->category_id)
            ->orderBy('items.id', 'desc')
            ->withCount(['reviews as rating' => function ($query) {
                $query->select(DB::raw('round(avg(rating),1)'));
            }])
            ->with(['wishlist' => function ($q) use ($request) {
                $q->where('reseller_id', $request->reseller_id);
            }])
            ->with('gallery')
            ->groupBy('items.id')
            ->paginate((($request->per_page) ? $request->per_page : 12));
        return ['products' => $query->items(), 'total_page' => (int)ceil($query->total() / $query->perPage())];
    }
    /**
     * 
     * @author SOORYA
     * 
     */
    public function getDraftProductList($request)
    {
        $query =  self::where('is_draft', null)->with('vendor')->where('deleted_at', null);
        if (isset($request->category_id) && isset($request->category_id)) {
            $query->where('category_id', $request->category_id);
        }

        if (isset($request->search) && isset($request->search)) {
            $query->where('name', 'LIKE', '%' . $request->search . '%');
        }
        if (isset($request->sort_by) && isset($request->sort_by)) {
            $query->orderBy($request->sort_by);
        }
        return $query->paginate($request->limit);
    }

    /**
     * 
     * @author SOORYA
     * 
     */
    public function getUnverifiedProductList($request)
    {
        $query =  self::where('is_verified', null)->with('vendor')->where('deleted_at', null);
        if (isset($request->category_id) && isset($request->category_id)) {
            $query->where('category_id', $request->category_id);
        }

        if (isset($request->search) && isset($request->search)) {
            $query->where('name', 'LIKE', '%' . $request->search . '%');
        }
        if (isset($request->sort_by) && isset($request->sort_by)) {
            $query->orderBy($request->sort_by);
        }
        return $query->paginate($request->limit);
    }

    /**
     * 
     * @author SOORYA
     */
    public function UnverifiedChangeApproveStatus($id)
    {
        $product =  self::where('id', $id)->first();
        if ($product->is_verified == null) {
            return self::where('id', $id)->update(['is_verified' => Carbon::now()->format('Y-m-d H:i:s')]);
        }
    }
    /**
     * 
     */
    public function getdraftProductsByVendorId($vendor_id)
    {
        return self::with('vendor', 'category', 'product_type', 'item_image')
            ->where('is_draft', null)
            ->where('vendor_id', $vendor_id)
            ->get();
    }

    /**
     * 
     */
    public function getvendorProductListByVendorId($vendor_id)
    {
        return self::with('vendor', 'item_image', 'categories:id,name', 'product_types:id,name')
            ->where('is_verified', '!=', null)
            ->where('vendor_id', $vendor_id)
            ->get();
    }

    /**
     * 
     * 
     */
    public function getFeaturedProducts($product_type_id, $limit)
    {
        return self::join('item_product_type', 'item_product_type.item_id', '=', 'items.id')
            ->where('item_product_type.product_type_id', $product_type_id)
            ->select('items.*', 'item_product_type.product_type_id')
            ->limit($limit)
            ->get();
    }

    /**
     * similar products
     * 
     */
    public function getSimilarProducts($id)
    {
        $item = self::find($id);
        $categoryIds = $item->categories->pluck('id')->toArray();
        $similarItems = Item::whereHas('categories', function ($query) use ($categoryIds) {
            $query->whereIn('category_id', $categoryIds);
        })
            ->where('id', '!=', $id)
            ->limit(8)
            ->get();
        return $similarItems;
    }

    /**
     * 
     * 
     */
    public function getItemWithDetails($slug)
    {
        return self::with('item_image', 'vendor')
            ->where('slug', $slug)
            ->first();
    }

    /**
     * 
     * 
     */
    public function getListItem()
    {
        return self::whereNull('parent_id')->get();
    }

    /**
     * 
     * 
     */
    public function getVariants($item_id)
    {
        $childItems = self::where('parent_id', $item_id)->with('item_image')->get();
        if ($childItems->count() > 0) {
            $parentItem = self::where('id', $item_id)->with('item_image')->get();
            return $parentItem->merge($childItems);
        } else {
            return self::where('id', $item_id)->with('item_image')->get();
        }
    }
}
