<?php

namespace App\Models;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;
use App\Models\DeliveryBoyEarnedHistory;
use App\Models\RestaurantEarnedHistory;
use Auth;
use DB;
class Order extends Model
{
    use SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'uid',
        'reseller_id',
        'invoice_number',
        'address',
        'total_quantity',
        'total_cost',
        'coupon_id',
        'discount_cost',
        'shipping_cost',
        'shipping_cost_info',
        'order_status_id',
        'payment_option_id',
        'payment_id',
        'invoice_pdf_src',
        'created_at',
        'grand_total',
        'commission'
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'updated_at','deleted_at'
    ];

    // public function getCreatedAtAttribute($value)
    // {
    //     return date('d-m-Y',strtotime($value));
    // }


    /**
    * 
    * @return void
    */
    public function products()
    {
        return $this->hasMany('App\Models\OrderDetail');
    }
    /**
    * 
    * @return void
    */
    public function product()
    {
        return $this->hasOne('App\Models\OrderDetail');
    }
    /**
    * 
    * @return void
    */
    public function vendor()
    {
        return $this->belongsTo('App\Models\Vendor','vendor_id','id');
    }

        /**
    * 
    * @return void
    */
    public function reseller()
    {
        return $this->belongsTo('App\Models\Reseller','reseller_id','id');
    }


    /**
    * 
    * @return void
    */
    public function status()
    {
        return $this->belongsTo('App\Models\OrderStatus','order_status_id','id');
    }

        /**
    * 
    * @return void
    */
    public function paymentOption()
    {
        return $this->belongsTo('App\Models\PaymentOption');
    }

     /**
    * 
    * @return void
    */
    public function route()
    {
        return $this->belongsTo('App\Models\Route');
    }

    public function getShippingCostInfoAttribute($value)
    {
        return json_decode($value,true);
    }

    /**
     * 
     * @return void
     * 
     */
    public function orderItems()
    {
        return $this->hasMany('App\Models\OrderDetail')->with('item:id,name');
    }
    /**
     *  Setup model event hooks
     */
    public static function boot()
    {
         parent::boot();
         self::creating(function ($model) {
             $model->uid = (string) Str::uuid();
         });
    }


    /**
    *
    *
    */
    public function getLastOrder()
    {
        $last_order = self::latest()->first();
        return ( $last_order != null  ) ? $last_order->id : 1;
    }



    /**
    *
    *
    */
    public function createOrder($input)
    {
        return self::create($input);
    }

    /**
    *
    *
    */
    public function updateOrder($id, $input)
    {
        return self::where('id',$id)->update($input);
    }
    /**
     * 
     * 
     */
    public function getResellerOrders($reseller_id)
    {
        return self::with('reseller','status','paymentOption','products.item')
                    ->where('reseller_id',$reseller_id)
                    ->orderBy('id','desc')
                    ->get();
    }
    /**
     * 
     * 
     */
    public function getSellerOrders($vendor_id)
    {
        return self::select(
        'orders.id',
        'orders.address',
        'orders.invoice_number',
        'orders.payment_option_id',
        'orders.created_at',
        'orders.created_at',
        DB::raw('SUM(order_details.net_total) AS vendor_bill_amount')
        )
        ->join('order_details', 'orders.id', '=', 'order_details.order_id')
        ->where('order_details.vendor_id',$vendor_id)
        ->orderBy('orders.id','desc')
        ->groupBy('order_details.order_id')
        ->with('paymentOption')
        ->get();


    }
     /**
     * 
     * 
     */
    public function getResellerOrderDetails($request)
    {
        return self::with('reseller', 'paymentOption')
                   ->whereHas('products', function($query) use($request) {
                    $query->where('reseller_id', $request->reseller_id); })
                    ->with(['products' => function($query) use($request) { 
                        $query->with('item','vendor','status')
                        ->where('reseller_id', $request->reseller_id); }])
                   ->where('id',$request->order_id)
                   ->first();
    }
    /**
     * 
     * 
     */
    public function getSellerOrderDetails($request)
    {
        return self::select(
        'orders.id',
        'orders.address',
        'orders.invoice_number',
        'orders.payment_option_id',
        'orders.created_at',
        'orders.created_at',
        DB::raw('SUM(order_details.total_cost) AS total_cost'),
        DB::raw('SUM(order_details.shipping_cost) AS vendor_shipping_cost'),
        DB::raw('SUM(order_details.net_total) AS vendor_bill_amount')
        )
        ->where('order_details.vendor_id',$request->vendor_id)
        ->join('order_details', 'orders.id', '=', 'order_details.order_id')
        ->with(['products' => function($query) use($request) { 
            $query->with('item','vendor','status')
            ->where('vendor_id', $request->vendor_id); }])
        ->with('paymentOption')
        ->where('orders.id',$request->order_id)
        ->groupBy('order_details.order_id')
        ->first();
}
    /**
    *
    *
    */
    public function getOrderByUid($uid)
    {
        return self::with('reseller','status','paymentOption','products.item')->where('uid',$uid)->first();
    }
    /**
     * 
     * @author PMS
     * 
     */
    public function getOrderById($id)
    {
        return self::select(
            'id',
            'uid',
            'invoice_number',
            'created_at as placed_time',
            'delivered_by',
            'total_quantity',
            'grand_total',
            'store_total_cost'
        )
        ->where('id',$id)
        ->with('orderItems:item_id,quantity,total_cost,store_cost,total_store_cost,order_id')
        ->with('deliveredBy:id,name,phone')
        ->first();

    }

     /**
    *
    *
    */
    public function getOrderByUserId($id)
    {
        return self::select(
            'id',
            'uid',
            'invoice_number',
            'created_at as placed_time',
            'order_status_id',
            'total_quantity',
            'coupon_id',
            'store_total_cost',
            'total_cost',
            'discount_cost',
            'shipping_cost as delivery_charge',
            'tax',
            'packing_charge',
            'rainy_rush',
            'grand_total',
            'commission',
            'is_rainy_rush',
            'grand_total',
            'address',
            'store_id',
            'created_at',
            'store_accepted_at',
            'out_for_delivery_at',
            'delivered_at',
            'delivered_by'
        )
        ->where('id',$id)
        ->with('store:id,name,short_address')
        ->with('orderItems:item_id,quantity,total_cost,order_id')
        ->with('orderItems:item_id,quantity,total_cost,order_id')
        ->with('deliveredBy:id,name,phone')
        ->first();
    }
    /**
     * 
     * 
     */
    public function getOrdersDetailsForDeliveryBoy($id)
    {
        return self::select(
            'id',
            'uid',
            'invoice_number',
            'created_at as placed_time',
            'order_status_id',
            'total_quantity',
            'shipping_cost',
            'commission',
            'grand_total',
            'address',
            'store_id'
        )
        ->where('id',$id)
        ->with('store:id,name,address,phone,latitude,longitude')
        ->with('orderItems:item_id,quantity,cost,total_cost,order_id')
        ->with('status:id,name,delivery_boy')
        ->first();
    }

    /**
    *
    *
    */
    public function getUserOrders($user_id)
    {
       return self::where('user_id',$user_id)
       ->select('id','user_id','invoice_number','address','total_cost','shipping_cost','commission','grand_total','order_status_id','payment_option_id')
       ->orderBy('id','DESC')
       ->get();
    }

    /**
    *
    *
    */
    public function getUserDeliveredOrdersHistory($user_id,$status,$limit,$offset)
    {
        
        return   self::select(
            'id',
            'user_id',
            'order_status_id',
            'invoice_number',
            'total_quantity',
            'grand_total',
            'store_id',
            'address',
            'created_at'
            )
        ->orderBy('created_at','desc')
        ->where('user_id',$user_id)
        ->whereIn('order_status_id',$status)
        ->offset(($offset) ? $offset :0)
        ->limit(($limit) ? $limit :5)
        ->with('store:id,name,address as store_address,logo')
        ->withCount('orderItems')
        ->with('status:id,name,user')
        ->get();
    }

    public function getUserDeliveredOrdersHistoryTotalpage($user_id,$status,$limit)
    {
        $result =  self::select('id','user_id','order_status_id','invoice_number','total_quantity','grand_total','store_id','address')
                        ->where('user_id',$user_id)
                        ->whereIn('order_status_id',$status)
                        ->with('store:id,name,address as store_address')
                        ->with('status:id,name,user')
                        ->paginate(($limit) ? $limit :5);
                        $total_page  = (int)ceil($result->total()/$result->perPage());
                        return $total_page;
    }

    /**
    *
    *
    */
    public function getUserOrderById($id)
    {
       return self::find($id);
    }

    /**
    *
    * @author SOORYA
    */
    public function getUserUpdateOrderById($id)
    {
        return self::whereHas('product', function($query) use($id) {
                        $query->where('id', $id); })
                    ->first();
    }

    /**
    *
    * @author SOORYA
    */    
    public function getVendorOrderById($id)
    {
       return self::whereHas('product', function($query) use($id) {
                        $query->where('id', $id); })
                    ->first();
    }

    public function updateUnassignedOrder($order_id,$data)
    {
        return self::where('id',$order_id)
        ->update($data);
    }

    /**
     * 
     * 
     */
    public function getOrderByOrderId($id)
    {
       return self::find($id);
    }

    /**
    *
    *
    */
    public function getStoreOrdersList($request)
    {

        $query = self::with('user','status','paymentOption','route','deliveredBy');

        if(!$request->is('admin'))
        {
            $query->where('store_id',Auth::user()->id);
        }
        

         if( isset($request->user_id) && isset($request->user_id))
         {
             $query->where('user_id',$request->user_id);
         }

         if( isset($request->order_status_id) && isset($request->order_status_id))
         {
             $query->where('order_status_id',$request->order_status_id);
         }


         if( isset($request->search) && isset($request->search))
         {
             $query->where('invoice_number','LIKE','%'.$request->search.'%');
         }
         return          $result = $query->latest()->paginate($request->limit);

    }

     /**
    *
    *
    */
    public function getAdminOrdersList($request)
    {

        $query = self::with('reseller','status','paymentOption');

         if( isset($request->reseller_id) && isset($request->reseller_id))
         {
             $query->where('reseller_id',$request->reseller_id);
         }

         if( isset($request->order_status_id) && isset($request->order_status_id))
         {
             $query->where('order_status_id',$request->order_status_id);
         }


         if( isset($request->search) && isset($request->search))
         {
             $query->where('invoice_number','LIKE','%'.$request->search.'%');
         }
         return  $result = $query->latest()->paginate($request->limit);

    }

   

    public function getDeliveryManagerOrdersList($request)
    {

        $query = self::with('store','user','status','paymentOption','deliveredBy');

        

         if( isset($request->user_id) && isset($request->user_id))
         {
             $query->where('user_id',$request->user_id);
         }

         if( isset($request->order_status_id) && isset($request->order_status_id))
         {
             $query->where('order_status_id',$request->order_status_id);
         }


         if( isset($request->search) && isset($request->search))
         {
             $query->where('invoice_number','LIKE','%'.$request->search.'%');
         }
         return          $result = $query->latest()->paginate($request->limit);

    }

    public function getStoreManagerOrdersList($request)
    {

        $query = self::with('store','user','status','paymentOption','deliveredBy');

        

         if( isset($request->user_id) && isset($request->user_id))
         {
             $query->where('user_id',$request->user_id);
         }

         if( isset($request->order_status_id) && isset($request->order_status_id))
         {
             $query->where('order_status_id',$request->order_status_id);
         }


         if( isset($request->search) && isset($request->search))
         {
             $query->where('invoice_number','LIKE','%'.$request->search.'%');
         }
         return          $result = $query->latest()->paginate($request->limit);

    }

    public function getStoreTotalRevenue($store_id)
    {
        return self::whereNotIn('order_status_id',[8,9])->where('store_id',$store_id)->sum('total_cost');
    }

    public function getStoreTotalOrders($store_id)
    {
        return self::whereNotIn('order_status_id',[8,9])->where('store_id',$store_id)->count();
    }


    public function getStoreTotalOrdersByPaymentOption($store_id)
    {
        return self::with('PaymentOption:name,id')->select('payment_option_id',DB::raw("COUNT(id) as orders"))->where('store_id',$store_id)->groupBy('payment_option_id')
        ->whereNotIn('order_status_id',[8,9])->get();
    }

     public function getStoreLast15DaysOrders($store_id)
    {
        return self::select( DB::raw('DATE(created_at) as date'),DB::raw('COUNT(id) as total'),DB::raw('SUM(total_cost) as total_cost') )
        ->where('store_id',$store_id)
        ->whereNotIn('order_status_id',[8,9])
        ->whereDate('created_at','>=',date('Y-m-d',strtotime('-15 days')) )
        ->groupBy('date')->get();
    }




    public function getAdminTotalRevenue()
    {
        return self::whereNotIn('order_status_id',[2,4])->sum('total_cost');
    }

    public function getTodayRevenue()
    {
        return self::whereNotIn('order_status_id',[2,4])->where('created_at','>',Carbon::now()->toDateString())->sum('total_cost');
    }

    public function getYesterdayRevenue()
    {
        return self::whereNotIn('order_status_id',[2,4])->where('created_at', '>=', Carbon::yesterday()->startOfDay())->where('created_at', '<=', Carbon::yesterday()->endOfDay())->sum('total_cost'); 
    }

    public function getLast7DaysRevenue()
    {
        return self::whereNotIn('order_status_id',[2,4])->where('created_at','>',Carbon::now()->subDays(7))->sum('total_cost'); 
    }

    public function getLastMonthRevenue()
    {
        return self::whereNotIn('order_status_id',[2,4]) ->whereDate('created_at','>=',date('Y-m-d',strtotime('first day of last month')) )
        ->whereDate('created_at','<=',date('Y-m-d',strtotime('last day of last month')) )->sum('total_cost'); 
    }

    public function getAdminTotalOrders()
    {
        return self::whereNotIn('order_status_id',[2,6])->count();
    }

    public function getTodayOrders()
    {
        return self::whereNotIn('order_status_id',[2,4])->where('created_at','>',Carbon::now()->toDateString())->count();
    }

    public function getYesterdayOrders()
    {
        return self::whereNotIn('order_status_id',[2,4])->where('created_at', '>=', Carbon::yesterday()->startOfDay())->where('created_at', '<=', Carbon::yesterday()->endOfDay())->count();
    }

    public function getLast7DaysOrders()
    {
        return self::whereNotIn('order_status_id',[2,4])->where('created_at','>',Carbon::now()->subDays(7))->count();
    }

    public function getLastMonthOrders()
    {
        return self::whereNotIn('order_status_id',[2,4]) ->whereDate('created_at','>=',date('Y-m-d',strtotime('first day of last month')) )
        ->whereDate('created_at','<=',date('Y-m-d',strtotime('last day of last month')) )->count();
    }

    public function getAdminTotalOrdersByPaymentOption()
    {
        return self::select('payment_option_id',DB::raw("COUNT(id) as orders"))
                    ->with('PaymentOption:name,id')
                    ->whereHas('products', function($query){
                        $query->whereNotIn('order_status_id',[2,6,7]);})
                    ->groupBy('payment_option_id')->get();
    }

     public function getAdminLast15DaysOrders()
    {
        return self::select( DB::raw('DATE(created_at) as date'),DB::raw('COUNT(id) as total'),DB::raw('SUM(total_cost) as total_cost') )
        ->whereNotIn('order_status_id',[2,4])
        ->whereDate('created_at','>=',date('Y-m-d',strtotime('-15 days')) )
        ->groupBy('date')->get();
    }

    public function getAdminTodayOrders()
    {
        return self::select( DB::raw('DATE(created_at) as date'),DB::raw('COUNT(id) as total'),DB::raw('SUM(total_cost) as total_cost') )
        ->whereNotIn('order_status_id',[2,4])
        ->whereDate('created_at','>=',date('Y-m-d',strtotime('today GMT')) )
        ->groupBy('date')->get();
    }
    
    public function getAdminYesterdayOrders()
    {
        return self::select( DB::raw('DATE(created_at) as date'),DB::raw('COUNT(id) as total'),DB::raw('SUM(total_cost) as total_cost') )
        ->whereNotIn('order_status_id',[2,4])
        ->whereDate('created_at', '>=', Carbon::yesterday()->startOfDay())->where('created_at', '<=', Carbon::yesterday()->endOfDay())
        ->groupBy('date')->get();
    }

    public function getAdminLast7DaysOrders()
    {
        return self::select( DB::raw('DATE(created_at) as date'),DB::raw('COUNT(id) as total'),DB::raw('SUM(total_cost) as total_cost') )
        ->whereNotIn('order_status_id',[2,4])
        ->whereDate('created_at','>=',date('Y-m-d',strtotime('-7 days')) )
        ->groupBy('date')->get();
    }

    public function getAdminLastMonthOrders()
    {
        return self::select( DB::raw('DATE(created_at) as date'),DB::raw('COUNT(id) as total'),DB::raw('SUM(total_cost) as total_cost') )
        ->whereNotIn('order_status_id',[2,4])
        ->whereDate('created_at','>=',date('Y-m-d',strtotime('first day of last month')) )
        ->whereDate('created_at','<=',date('Y-m-d',strtotime('last day of last month')) )
        ->groupBy('date')->get();
    }

    public function getDeliverOrders($delivered_by)
    {
        return self::where('delivered_by',$delivered_by)
               ->whereIn('order_status_id',[2,3,4,5])
               ->orderBy('id','desc')
               ->get();
    }

    public function getDeliveredOrders($delivered_by,$limit)
    {
       $result      =  self::where('delivered_by',$delivered_by)
                                ->whereIn('order_status_id',[6])
                                ->orderBy('id','desc')
                                ->paginate( (($limit) ? $limit : 5) );
       
       return [ 'orders' => $result->items() , 'total_pages' => (int)ceil($result->total()/$result->perPage())];
       
    }

    /**
     * 
     * @author PMS
     * 
     */
    public function fetchOrderCount($store_id,$order_status,$date)
    {
        return self::where('store_id',$store_id)
        ->whereIn('order_status_id',$order_status)
        ->whereDate('created_at',$date)
        ->get()
        ->Count();
    }

    /**
     * 
     * @author PMS
     * 
     */
  
    public function getUserOrdersList($user_id)
    {

        $query =  self::where("order_status_id",4)
                      ->where("user_id",$user_id)
                      ->with('paymentOption:id,name','status:id,name');
                       
      
        return $query->paginate(10);
    }
    
    /**
     * 
     * @author PMS
     * store total order in between dates
     */
    public function getTotalStoreOrdersBetweenDates($store_id,$start_date,$end_date)
    {
        return self::select('id')
        ->where('store_id',$store_id)
        ->where('order_status_id',4)
        ->whereDate('created_at','>=',$start_date)
        ->whereDate('created_at','<=',$end_date)
        ->get()->count();
    }

 


    public function  getGrandTotal($store_id)
    {
        return self::select(DB::raw('sum(grand_total) as total'))
                    ->where('store_id',$store_id)
                    ->groupBy('store_id')
                    ->where('order_status_id',4)
                    ->pluck('total');           
    }

    public function getTodayGrandTotal($store_id)
    {
        return self::select(DB::raw('sum(grand_total) as total'))
                    ->where('store_id',$store_id)
                    ->whereDate('created_at','>=',date('Y-m-d',strtotime('today GMT')))
                    ->groupBy('store_id')
                    ->where('order_status_id',4)
                    ->pluck('total');
                   
                    
    }     

    public function getYesterdayGrandTotal($store_id)
    {
        return self::select(DB::raw('sum(grand_total) as total'))
                    ->where('store_id',$store_id)
                    ->whereDate('created_at', '>=', Carbon::yesterday()->startOfDay())
                    ->where('created_at', '<=', Carbon::yesterday()->endOfDay())
                    ->where('order_status_id',4)
                    ->pluck('total');
                   
    }

    public function getLast7DaysGrandTotal($store_id)
    {
        return self::select(DB::raw('sum(grand_total) as total'))
                    ->where('store_id',$store_id)
                    ->whereDate('created_at','>=',date('Y-m-d',strtotime('-7 days')))
                    ->groupBy('store_id')->where('order_status_id',4)
                    ->pluck('total');
    
    }

    public function getLastMonthGrandTotal($store_id)
    {
        return self::select(DB::raw('sum(grand_total) as total'))
                    ->where('store_id',$store_id)
                    ->whereDate('created_at','>=',date('Y-m-d',strtotime('first day of last month')))
                    ->whereDate('created_at','<=',date('Y-m-d',strtotime('last day of last month')))
                    ->groupBy('store_id')
                    ->where('order_status_id',4)
                    ->pluck('total');
                   
    }


   
    public function getOrderCount($store_id)
    {
        return self::where('store_id',$store_id)->where('order_status_id',4)->count();
                   
    }

    public function getTodayOrderCount($store_id)
    {
        return self::where('store_id',$store_id)
                    ->where('order_status_id',4)
                    ->whereDate('created_at','>=',date('Y-m-d',strtotime('today GMT')))
                    ->count();            
    }

    public function getYesterdayOrderCount($store_id)
    {
        return self::where('store_id',$store_id)
                    ->where('order_status_id',4)
                    ->whereDate('created_at', '>=', Carbon::yesterday()->startOfDay())->where('created_at', '<=', Carbon::yesterday()->endOfDay())
                    ->count();
                   
    }

    public function getLast4DaysOrderCount($store_id)
    {
        return self::where('store_id',$store_id)
                    ->where('order_status_id',4)   
                    ->whereDate('created_at','>=',date('Y-m-d',strtotime('-7 days')))
                    ->count();
                   
    }

    public function getLastMonthOrderCount($store_id)
    {
        return self::where('store_id',$store_id)
                    ->where('order_status_id',4)
                    ->whereDate('created_at','>=',date('Y-m-d',strtotime('first day of last month')) )
                    ->whereDate('created_at','<=',date('Y-m-d',strtotime('last day of last month')) )
                    ->count();
                   
    }

    public function getOrders($from,$to)
    {
        $orders = self::whereBetween('created_at', [$from ,$to])->where('order_status_id',4)
                    ->orderBy('id','ASC')
                    ->get();
        if(count($orders) > 0)
        {
            $result                    = [];
            foreach($orders as $order)
            {
             
                $result[]= [
                    'invoice_number'        => $order->invoice_number,
                    'created_at'            => $order->created_at,
                    'grand_total'           => $order->total_cost,
    
                ];
            }
            return ['orders' => $result];
        }
        else
        {
            return [];
        }
    }

    public function getDelayOrders()
    {
        return self::whereIn('order_status_id',[1,3,5,6])->get();
    }

    /**
     * @author PMS
     */
    public function fetchOrderDetailsForTracking($order_id,$status)
    {
        return self::select(
        'id',
        'store_id',
        'user_id',
        'invoice_number',
        'latitude',
        'longitude',
        'delivered_by',
        'order_status_id',
        'delivery_boy_accepted_at',
        'delivery_boy_picked_at',
        'out_for_delivery_at'
        )
        ->where('id',$order_id)
        ->whereIn('order_status_id',$status)
        ->with('store:id,name,address,phone,latitude,longitude')
        ->with('deliveredBy:id,name,phone,service_location_id')
        ->with('status:id,user')
        ->first();
    }
    /**
     * 
     */
    public function fetchOrderById($id)
    {
        return self::find($id);
    }

    /**
     * 
     */
    public function getDeliveryBoyOrderByDate($delivery_boy_id,$date)
    {
        return self::where('delivered_by',$delivery_boy_id)
        ->select(
            'id',
            'user_id',
            'store_id',
            'invoice_number',
            'address',
            'total_cost',
            'shipping_cost',
            'commission',
            'grand_total',
            'order_status_id',
            'payment_option_id',
            'delivered_by',
            'created_at'
            )
        ->whereDate('created_at',$date)
        ->with('status:id,name,delivery_boy')
        ->with('store:id,name,address,phone,latitude,longitude')
        ->with('PaymentOption:name,id')
        ->get();
    }
    
    /**
     * 
     */
    public function getStoreOrderByDate($store_id,$date)
    {
        return self::where('store_id',$store_id)
        ->select(
            'id',
            'user_id',
            'store_id',
            'invoice_number',
            'address',
            'total_cost',
            'shipping_cost',
            'commission',
            'grand_total',
            'order_status_id',
            'payment_option_id',
            'delivered_by',
            'created_at'
            )
        ->whereDate('created_at',$date)
        ->with('status:id,name,store')
        ->with('deliveredBy:id,name,phone,service_location_id')
        ->with('PaymentOption:name,id')
        ->get();
    }

    
}