<?php

namespace App\Models;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use DB;
class OrderDetail extends Model
{
    use SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'item_id',
        'quantity',
        'order_id',
        'vendor_id',
        'reseller_id',
        'cost',
        'total_cost',
        'shipping_cost',
        'net_total',
        'store_cost',
        'total_store_cost',
        'order_status_id',
        'delivery_partner_name',
        'link',
        'order_tracking_id',
        'cancellation_reason'
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    public $timestamps = true;

    protected $hidden  = [
        'created_at','updated_at','deleted_at'
    ];

    /**
    * 
    * @return void
    */
    public function vendor()
    {
        return $this->belongsTo('App\Models\Vendor','vendor_id','id');
    }

    /**
    * 
    * @return void
    */
    public function reseller()
    {
        return $this->belongsTo('App\Models\Reseller','reseller_id','id');
    }
    
    /**
    * 
    * @return void
    */
    public function order()
    {
        return $this->belongsTo('App\Models\Order','order_id','id');
    }

    /**
    * 
    * @return void
    */
    public function item()
    {
        return $this->belongsTo('App\Models\Item','item_id','id');
    }

    /**
    * 
    * @return void
    */
    public function status()
    {
        return $this->belongsTo('App\Models\OrderStatus','order_status_id','id');
    }

    /**
    *
    *
    */
    public function getOrderProduct($id)
    {
       return self::find($id);
    }

    /**
    *
    *
    */
    public function getResellerOrders($id)
    {
       return self::with('reseller','status')
                    ->where('order_id',$id)
                    ->get();
    }

    /**
    *
    *
    */
    public function getVendorOrders($id)
    {
       return self::with('vendor','status')
                    ->where('order_id',$id)
                    ->get();
    }

    /**
    *
    *
    */
    public function createOrderDetails($input)
    {
        return self::insert($input);
    }
    /**
    *
    * @author SOORYA
    */
    public function getMostVendorOrders()
    {
       return self::select('vendor_id', DB::raw('count(order_id) as orders'))
                    ->where('order_status_id', 5)
                    ->orderBy('orders','DESC')
                    ->groupBy('vendor_id')
                    ->limit(5)->get();
    }

    public function getMostResellerOrders()
    {
       return self::select('reseller_id', DB::raw('count(order_id) as orders'))
                    ->where('order_status_id',5)
                    ->orderBy('orders','DESC')
                    ->groupBy('reseller_id')
                    ->limit(5)->get();
    }

    /**
     * 
     * @author SOORYA
     */
    public function getOrderProductById($order_id, $product_id)
    {
       return self::where('order_id', $order_id)->where('item_id', $product_id)->first();
    }

    /**
     * 
     * @author SOORYA
     */
    public function updateOrder($id, $input)
    {
        return self::where('id',$id)->update($input);
    }

    public function getProductOrders($product_id)
    {
        return self::where('item_id',$product_id)->sum('quantity');
    }

    public function getTodayProductOrders($product_id)
    {
        return self::where('item_id',$product_id)->where('created_at','>',Carbon::now()->toDateString())->sum('quantity');
    }

    public function getYesterdayProductOrders($product_id)
    {
        return self::where('item_id',$product_id)->where('created_at', '>=', Carbon::yesterday()->startOfDay())->where('created_at', '<=', Carbon::yesterday()->endOfDay())->sum('quantity');
    }

    public function getLast7DaysProductOrders($product_id)
    {
        return self::where('item_id',$product_id)->where('created_at','>',Carbon::now()->subDays(7))->sum('quantity');
    }

    public function getLastMonthProductOrders($product_id)
    {
        return self::where('item_id',$product_id) ->whereDate('created_at','>=',date('Y-m-d',strtotime('first day of last month')) )
        ->whereDate('created_at','<=',date('Y-m-d',strtotime('last day of last month')) )->sum('quantity');
    }

    public function getAdminTotalRevenue()
    {
        return self::whereNotIn('order_status_id',[2,6,7])->sum('total_cost');
    }

    public function getAdminTotalOrders()
    {
        return self::whereNotIn('order_status_id',[2,6,7])->count();
    }

    public function getAdminLast15DaysOrders()
    {
        return self::select( DB::raw('DATE(created_at) as date'),DB::raw('COUNT(id) as total'),DB::raw('SUM(total_cost) as total_cost') )
        ->whereNotIn('order_status_id',[2,6,7])
        ->whereDate('created_at','>=',date('Y-m-d',strtotime('-15 days')) )
        ->groupBy('date')->get();
    }

}
